import redis
from pydantic import BaseModel
from typing import Type, TypeVar, Generic, Dict, Any, Iterator, Tuple, List, Union, Literal
import json

T = TypeVar('T', bound=BaseModel)
K = TypeVar('K', str, int)

class ObjectStorageUnit(Generic[T, K]):
    def __init__(self, prefix: str, model: Type[T], expire: int = None):
        self.model = model
        if not issubclass(model, BaseModel):
            raise TypeError("Model must be a subclass of pydantic.BaseModel")
        self._prefix = prefix
        self._expire = expire
        self._redis_client: redis.Redis = None

    def set_redis_client(self, redis_client: redis.Redis, default_expire: int | None):
        self._redis_client = redis_client
        if self._expire is None:
            self._expire = default_expire

    def _check_redis_client(self):
        if self._redis_client is None:
            raise ValueError("Redis client is not set")

    def _format_key(self, key: K) -> str:
        return f"{self._prefix}_{str(key)}"

    def _parse_key(self, key_str: str) -> K:
        key_part = key_str[len(self._prefix) + 1:]
        if isinstance(self.key_type, type) and self.key_type is int:
            return int(key_part)  # type: ignore
        return key_part  # type: ignore

    def __setitem__(self, key: K, value: T):
        self._check_redis_client()
        if not isinstance(value, BaseModel):
            raise TypeError("Value must be an instance of pydantic.BaseModel")

        redis_key = self._format_key(key)
        self._redis_client.set(redis_key, value.model_dump_json(), ex=self._expire)

    def __getitem__(self, key: K) -> T | None:
        self._check_redis_client()
        redis_key = self._format_key(key)
        data = self._redis_client.get(redis_key)
        if data is None:
            return None
        return self.model.model_validate_json(data.decode())

    def __contains__(self, key: K) -> bool:
        self._check_redis_client()
        redis_key = self._format_key(key)
        return self._redis_client.exists(redis_key) > 0

    def get(self, key: K) -> T | None:
        return self.__getitem__(key)

    def update(self, key: K, value: T):
        self.__setitem__(key, value)

    def update_some_key(self, key: K, field: str, value: Any):
        obj = self.__getitem__(key)
        if obj is not None:
            setattr(obj, field, value)
            self.__setitem__(key, obj)

    def update_many_keys(self, key: K, values: Dict[str, Any]):
        obj = self.__getitem__(key)
        if obj is not None:
            for field, value in values.items():
                setattr(obj, field, value)
            self.__setitem__(key, obj)

    def remove(self, key: K):
        self._check_redis_client()
        redis_key = self._format_key(key)
        self._redis_client.delete(redis_key)

    def clear_all(self):
        self._check_redis_client()
        keys = self._redis_client.keys(f"{self._prefix}_*")
        for key in keys:
            self._redis_client.delete(key)

    def __iter__(self) -> Iterator[K]:
        return iter(self.keys())

    def keys(self) -> List[K]:
        self._check_redis_client()
        keys = self._redis_client.keys(f"{self._prefix}_*")
        return [self._parse_key(key.decode()) for key in keys]

    def items(self) -> Iterator[Tuple[K, T]]:
        self._check_redis_client()
        for key in self.keys():
            value = self[key]
            if value is not None:
                yield key, value

    def __len__(self) -> int:
        self._check_redis_client()
        return len(self._redis_client.keys(f"{self._prefix}_*"))

    @property
    def key_type(self):
        # Пытаемся получить информацию о типе ключа
        # Это работает только если вы создаёте экземпляр с конкретным типом, например: ObjectStorageUnit[MyModel, int]
        return self.__orig_class__.__args__[1]
