import redis.asyncio as redis
from pydantic import BaseModel
from typing import Type, TypeVar, Generic, Dict, Any
import json

T = TypeVar('T', bound=BaseModel)

class ObjectStorageUnitAsync(Generic[T]):
    def __init__(self, prefix: str, model: Type[T]):
        """
        Инициализация объекта ObjectStorageAsync.

        :param prefix: Префикс для ключей в Redis.
        :param model: Класс модели, наследуемый от pydantic.BaseModel.
        """
        self.model = model
        if not issubclass(model, BaseModel):
            raise TypeError("Model must be a subclass of pydantic.BaseModel")
        self.prefix = prefix
        self._redis_client: redis.Redis = None  # Инициализируем _redis_client как None

    async def __setitem__(self, key: int, value: T):
        """
        Сохраняет объект в Redis.

        :param key: Ключ для сохранения объекта.
        :param value: Объект, который нужно сохранить, должен быть экземпляром pydantic.BaseModel.
        """
        # Проверяем, что значение является экземпляром pydantic.BaseModel
        if not isinstance(value, BaseModel):
            raise TypeError("Value must be an instance of pydantic.BaseModel")
        # Сохраняем объект в Redis в формате JSON
        await self._redis_client.set(key, value.model_dump_json())

    async def __getitem__(self, key: int) -> T:
        """
        Получает объект из Redis по ключу.

        :param key: Ключ для получения объекта.
        :return: Объект модели, валидированный из JSON.
        :raises KeyError: Если ключ не найден в Redis.
        """
        # Получаем данные из Redis по ключу
        data = await self._redis_client.get(key)
        if data is None:
            raise KeyError(f"Key {key} not found")
        # Валидируем и возвращаем объект модели из JSON
        return self.model.model_validate_json(data.decode())

    async def get(self, key: int) -> T:
        """
        Получает объект по ключу, используя метод __getitem__.

        :param key: Ключ для получения объекта.
        :return: Объект модели.
        """
        # Получаем объект по ключу, используя метод __getitem__
        return await self.__getitem__(key)

    async def update(self, key: int, value: T):
        """
        Обновляет объект по ключу.

        :param key: Ключ для обновления объекта.
        :param value: Объект, который нужно обновить, должен быть экземпляром pydantic.BaseModel.
        """
        # Обновляем объект по ключу, используя метод __setitem__
        await self.__setitem__(key, value)

    async def update_some_key(self, key: int, field: str, value: Any):
        """
        Обновляет конкретное поле объекта по ключу.

        :param key: Ключ для обновления объекта.
        :param field: Поле, которое нужно обновить.
        :param value: Новое значение для поля.
        """
        # Обновляем конкретное поле объекта по ключу
        obj = await self.__getitem__(key)
        setattr(obj, field, value)
        await self.__setitem__(key, obj)

    async def update_many_keys(self, key: int, values: Dict[str, Any]):
        """
        Обновляет несколько полей объекта по ключу.

        :param key: Ключ для обновления объекта.
        :param values: Словарь с полями и их новыми значениями.
        """
        # Обновляем несколько полей объекта по ключу
        obj = await self.__getitem__(key)
        for field, value in values.items():
            setattr(obj, field, value)
        await self.__setitem__(key, obj)

    async def remove(self, key: int):
        """
        Удаляет объект из Redis по ключу.

        :param key: Ключ для удаления объекта.
        """
        # Удаляем объект из Redis по ключу
        await self._redis_client.delete(key)
