from typing import Any
import redis.asyncio as redis

from blackbot_redis_storage.asyncio.ObjectStorageUnitAsync import ObjectStorageUnitAsync

class RedisStorageAsync:
    def __init__(self, redis_host: str, redis_port: int, redis_password: str, redis_db: int = 0):
        """
        Инициализация подключения к Redis.

        :param redis_host: Хост Redis.
        :param redis_port: Порт Redis.
        :param redis_password: Пароль Redis.
        """
        self.redis_host = redis_host
        self.redis_port = redis_port
        self.redis_password = redis_password
        self._redis_client = redis.Redis(
            host=self.redis_host,
            port=self.redis_port,
            password=self.redis_password,
            db=redis_db,
        )
        self._initialize_storage_units()

    def _initialize_storage_units(self):
        for attr_name in dir(self):
            attr = getattr(self, attr_name)
            if isinstance(attr, ObjectStorageUnitAsync):
                attr._redis_client = self._redis_client

    def __setattr__(self, name: str, value: Any):
        if isinstance(value, ObjectStorageUnitAsync):
            value._redis_client = self._redis_client
        super().__setattr__(name, value)

    async def connected(self) -> bool:
        """
        Проверяет подключение к Redis.

        :return: True, если подключение успешно, иначе False.
        """
        try:
            await self._redis_client.ping()
            return True
        except redis.ConnectionError:
            return False

    def __init_subclass__(cls, **kwargs):
        super().__init_subclass__(**kwargs)
        # Удаляем вызов _initialize_storage_units из __init_subclass__

    def get_client(self) -> redis.Redis:
        """
        Возвращает клиент Redis.

        :return: Клиент Redis.
        """
        return self._redis_client
