import os
import pytest
import asyncio
from pydantic import BaseModel
from blackbot_redis_storage.asyncio import RedisStorageAsync, ObjectStorageUnitAsync
from pytest_asyncio import fixture


class UserModel(BaseModel):
    name: str
    age: int

class CustomRedisStorageAsync(RedisStorageAsync):
    users = ObjectStorageUnitAsync[UserModel](prefix='users', model=UserModel)

@fixture
async def setup_storage():
    """
    Настраивает подключение к Redis с использованием переменных окружения.
    """
    redis_host = os.getenv('REDIS_HOST')
    redis_port = int(os.getenv('REDIS_PORT'))
    redis_password = os.getenv('REDIS_PASSWORD')
    redis_db = os.getenv('REDIS_DB')
    storage = CustomRedisStorageAsync(redis_host, redis_port, redis_password, redis_db)
    return storage

@pytest.mark.asyncio
async def test_connection_async(setup_storage):
    """
    Тестирует подключение к Redis.
    """
    storage = setup_storage
    assert await storage.connected() is True

@pytest.mark.asyncio
async def test_set_and_get_user_async(setup_storage):
    """
    Тестирует установку и получение одного пользователя.
    """
    storage = setup_storage
    user = UserModel(name='John Doe', age=30)
    await storage.users.__setitem__(1, user)
    assert await storage.users.__getitem__(1) == user
    assert await storage.users.get(1) == user

@pytest.mark.asyncio
async def test_set_and_get_multiple_users_async(setup_storage):
    """
    Тестирует установку и получение нескольких пользователей.
    """
    storage = setup_storage
    user1 = UserModel(name='John Doe', age=30)
    user2 = UserModel(name='Jane Doe', age=25)
    await storage.users.__setitem__(1, user1)
    await storage.users.__setitem__(2, user2)
    assert await storage.users.__getitem__(1) == user1
    assert await storage.users.__getitem__(2) == user2
    assert await storage.users.__getitem__(1) != await storage.users.__getitem__(2)

@pytest.mark.asyncio
async def test_remove_user_async(setup_storage):
    """
    Тестирует удаление пользователя.
    """
    storage = setup_storage
    user = UserModel(name='John Doe', age=30)
    await storage.users.__setitem__(1, user)
    await storage.users.remove(1)
    with pytest.raises(KeyError):
        await storage.users.__getitem__(1)

@pytest.mark.asyncio
async def test_remove_nonexistent_user_async(setup_storage):
    """
    Тестирует удаление несуществующего пользователя.
    """
    storage = setup_storage
    await storage.users.remove(2)
    with pytest.raises(KeyError):
        await storage.users.__getitem__(2)

@pytest.mark.asyncio
async def test_update_user_async(setup_storage):
    """
    Тестирует обновление пользователя.
    """
    storage = setup_storage
    user = UserModel(name='John Doe', age=30)
    await storage.users.__setitem__(1, user)
    updated_user = UserModel(name='John Doe', age=31)
    await storage.users.update(1, updated_user)
    assert await storage.users.__getitem__(1) == updated_user

@pytest.mark.asyncio
async def test_update_some_key_async(setup_storage):
    """
    Тестирует обновление конкретного поля пользователя.
    """
    storage = setup_storage
    user = UserModel(name='John Doe', age=30)
    await storage.users.__setitem__(1, user)
    await storage.users.update_some_key(1, 'age', 31)
    assert (await storage.users.__getitem__(1)).age == 31

@pytest.mark.asyncio
async def test_update_many_keys_async(setup_storage):
    """
    Тестирует обновление нескольких полей пользователя.
    """
    storage = setup_storage
    user = UserModel(name='John Doe', age=30)
    await storage.users.__setitem__(1, user)
    await storage.users.update_many_keys(1, {'name': 'Jane Doe', 'age': 25})
    assert (await storage.users.__getitem__(1)).name == 'Jane Doe'
    assert (await storage.users.__getitem__(1)).age == 25
