import os
from pydantic import BaseModel
from blackbot_redis_storage.RedisStorage import RedisStorage
from blackbot_redis_storage.ObjectStorageUnit import ObjectStorageUnit

class UserModel(BaseModel):
    name: str
    age: int

class CustomRedisStorage(RedisStorage):
    users = ObjectStorageUnit[UserModel, int](prefix='users_key_int', model=UserModel)
    users_key_as_string = ObjectStorageUnit[UserModel, str](prefix='users_key_as_string', model=UserModel)
    always_empty_for_test = ObjectStorageUnit[UserModel, str](prefix='always_empty_for_test', model=UserModel)

def setup_storage():
    redis_host = os.getenv('REDIS_HOST')
    redis_port = int(os.getenv('REDIS_PORT'))
    redis_password = os.getenv('REDIS_PASSWORD')
    redis_db = os.getenv('REDIS_DB')
    return CustomRedisStorage(redis_host, redis_port, redis_password, redis_db)

def test_connection():
    storage = setup_storage()
    assert storage.connected() is True

def test_set_and_get_user():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users[1] = user
    assert storage.users[1] == user
    assert storage.users.get(1) == user

def test_set_and_get_multiple_users():
    storage = setup_storage()
    user1 = UserModel(name='John Doe', age=30)
    user2 = UserModel(name='Jane Doe', age=25)
    storage.users[1] = user1
    storage.users[2] = user2
    assert storage.users[1] == user1
    assert storage.users[2] == user2
    assert storage.users[1] != storage.users[2]

def test_remove_user():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users[1] = user
    storage.users.remove(1)
    assert storage.users[1] is None

def test_remove_nonexistent_user():
    storage = setup_storage()
    result = storage.users.remove(999)
    assert result is None

def test_update_user():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users[1] = user
    updated_user = UserModel(name='John Doe', age=31)
    storage.users.update(1, updated_user)
    assert storage.users[1] == updated_user

def test_update_some_key():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users[1] = user
    storage.users.update_some_key(1, 'age', 31)
    assert storage.users[1].age == 31

def test_update_many_keys():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users[1] = user
    storage.users.update_many_keys(1, {'name': 'Jane Doe', 'age': 25})
    assert storage.users[1].name == 'Jane Doe'
    assert storage.users[1].age == 25

def test_iterate_over_users():
    storage = setup_storage()
    storage.users.clear_all()  # Clear any existing data
    user1 = UserModel(name='John Doe', age=30)
    user2 = UserModel(name='Jane Doe', age=25)
    storage.users[1] = user1
    storage.users[2] = user2

    keys = list(storage.users)
    users = [storage.users[key] for key in keys]

    assert len(keys) == 2
    assert user1 in users
    assert user2 in users

def test_length_of_users():
    storage = setup_storage()
    storage.users.clear_all()  # Clear any existing data
    user1 = UserModel(name='John Doe', age=30)
    user2 = UserModel(name='Jane Doe', age=25)
    storage.users[1] = user1
    storage.users[2] = user2
    assert len(storage.users) == 2

def test_iterate_empty():
    storage = setup_storage()
    assert list(storage.always_empty_for_test) == []
    assert len(storage.always_empty_for_test) == 0

def test_set_and_get_user_by_name():
    storage = setup_storage()
    user = UserModel(name='John Doe', age=30)
    storage.users_key_as_string['John_Doe'] = user
    assert storage.users_key_as_string['John_Doe'] == user
    assert storage.users_key_as_string.get('John_Doe') == user

def test_keys_method():
    storage = setup_storage()
    user1 = UserModel(name='Alice', age=28)
    user2 = UserModel(name='Bob', age=35)
    storage.users[10] = user1
    storage.users[20] = user2
    keys = storage.users.keys()
    assert 10 in keys
    assert 20 in keys

def test_items_method():
    storage = setup_storage()
    user1 = UserModel(name='Alice', age=28)
    user2 = UserModel(name='Bob', age=35)
    storage.users[10] = user1
    storage.users[20] = user2
    items = dict(storage.users.items())
    assert items[10] == user1
    assert items[20] == user2
