from __future__ import annotations

# redis_storage/collections.py
from typing import Any, Generic, TypeVar, cast, Iterator

from redis import Redis

from .locking import Lock
from .types import BaseModel, Field, NestedModel


T = TypeVar("T", bound=BaseModel)
K = TypeVar("K")


# Этот класс является "маркером" для аннотаций типов,
# он не несет логики, но позволяет нам передать два типа (Модель, Ключ)
# в BaseRedisStorage.
class Collection(Generic[T, K]):
    # Мы определяем здесь методы, чтобы IDE знала, что они существуют.
    # Реальная логика находится в ModelCollection.
    def __getitem__(self, key: K) -> T: ...
    def __setitem__(self, key: K, model: T) -> None: ...
    def get(self, key: K, default: T | None = None) -> T | None: ...
    def remove(self, key: K) -> int: ...
    def exists(self, key: K) -> bool: ...
    def lock(self, key: K, ttl_ms: int) -> Lock: ...
    def items(self) -> Iterator[tuple[K, T]]: ...
    def keys(self) -> Iterator[K]: ...


class _RedisBackedProxy:
    """Mixin implementing Redis-backed attribute access for model fields."""

    def __init__(self, redis: Redis, key: str, model_cls: type[T], expire: int | None = None):
        # use object.__setattr__ to bypass overridden __setattr__ below
        object.__setattr__(self, "_redis", redis)
        object.__setattr__(self, "_key", key)
        object.__setattr__(self, "_model_cls", model_cls)
        object.__setattr__(self, "_expire", expire)

    def __getattribute__(self, name: str) -> Any:
        if name.startswith("_"):
            return object.__getattribute__(self, name)

        model_cls = cast(type, object.__getattribute__(self, "_model_cls"))
        field = getattr(model_cls, name, None)
        if isinstance(field, Field):
            # Submodel access returns a nested proxy bound to derived key
            if isinstance(field, NestedModel):
                sub_model_cls = field.model_cls
                if sub_model_cls is None:
                    raise TypeError(f"SubModel '{name}' is not configured with a model class")
                redis = object.__getattribute__(self, "_redis")
                key = object.__getattribute__(self, "_key")
                parent_expire = object.__getattribute__(self, "_expire")
                nested_key = f"{key}:{name}"
                # Prefer submodel's own Config TTL, then inherit from parent
                sub_expire = getattr(getattr(sub_model_cls, "Config", object), "default_expire", None)
                expire = sub_expire if sub_expire is not None else parent_expire
                nested_proxy_cls = type(
                    f"{sub_model_cls.__name__}Proxy",
                    (_NestedRedisBackedProxy, sub_model_cls),  # type: ignore[misc]
                    {},
                )
                return nested_proxy_cls(redis, nested_key, sub_model_cls, expire)

            redis = object.__getattribute__(self, "_redis")
            key = object.__getattribute__(self, "_key")
            raw = redis.hget(key, name)
            # Проверяем, существует ли ключ вообще, чтобы отличить None от отсутствия
            if raw is None and not redis.exists(key):
                raise KeyError(f"Object with key '{key}' not found.")
            return field.deserialize(raw)

        return object.__getattribute__(self, name)

    def __setattr__(self, name: str, value: Any) -> None:
        if name.startswith("_"):
            object.__setattr__(self, name, value)
            return

        model_cls = cast(type, object.__getattribute__(self, "_model_cls"))
        field = getattr(model_cls, name, None)
        if isinstance(field, Field):
            redis = object.__getattribute__(self, "_redis")
            key = object.__getattribute__(self, "_key")
            # Проверяем, существует ли ключ перед обновлением
            if not redis.exists(key):
                raise KeyError(f"Object with key '{key}' not found. Cannot set attribute on non-existent object.")
            serialized = field.serialize(value)
            redis.hset(key, name, serialized)
            return

        object.__setattr__(self, name, value)


class _NestedRedisBackedProxy(_RedisBackedProxy):
    """Proxy for nested submodels. Allows lazy creation and TTL application."""

    def __setattr__(self, name: str, value: Any) -> None:
        if name.startswith("_"):
            object.__setattr__(self, name, value)
            return

        model_cls = cast(type, object.__getattribute__(self, "_model_cls"))
        field = getattr(model_cls, name, None)
        if isinstance(field, Field):
            redis = object.__getattribute__(self, "_redis")
            key = object.__getattribute__(self, "_key")
            expire = object.__getattribute__(self, "_expire")
            serialized = field.serialize(value)
            redis.hset(key, name, serialized)
            if expire is not None:
                redis.expire(key, expire)
            return

        object.__setattr__(self, name, value)

class ModelCollection(Generic[T, K]):
    def __init__(self, redis: Redis, model_cls: type[T], key_type: type[K], prefix: str, expire: int | None):
        self.redis = redis
        self.model_cls = model_cls
        self.key_type = key_type
        self.prefix = prefix
        self.expire = expire

    def _make_key(self, id_: K) -> str:
        return f"{self.prefix}:{id_}"

    def __getitem__(self, key: K) -> T:
        # Dynamically build a proxy class that inherits from the model class
        # so IDEs know all model fields and methods.
        proxy_cls = type(
            f"{self.model_cls.__name__}Proxy",
            (_RedisBackedProxy, self.model_cls),  # type: ignore[misc]
            {},
        )
        redis_key = self._make_key(key)
        # Проверка существования ключа перед созданием прокси
        if not self.redis.exists(redis_key):
            raise KeyError(f"Object with key '{key}' not found.")
        return proxy_cls(self.redis, redis_key, self.model_cls, self.expire)

    def get(self, key: K, default: T | None = None) -> T | None:
        """Безопасно получить объект по ключу.

        Возвращает прокси-модель, если ключ существует в Redis, иначе
        возвращает `default` (по умолчанию None). Не изменяет TTL ключа.
        """
        proxy_cls = type(
            f"{self.model_cls.__name__}Proxy",
            (_RedisBackedProxy, self.model_cls),  # type: ignore[misc]
            {},
        )
        redis_key = self._make_key(key)
        if not self.redis.exists(redis_key):
            return default
        return proxy_cls(self.redis, redis_key, self.model_cls, self.expire)

    def __setitem__(self, key: K, model: T) -> None:
        data = model.to_redis_hash()
        redis_key = self._make_key(key)
        if data:
            self.redis.hset(redis_key, mapping=data)
        if self.expire is not None:
            self.redis.expire(redis_key, self.expire)

        # Persist submodels into separate hashes
        for field_name, field in self.model_cls._fields.items():
            if isinstance(field, NestedModel):
                sub_model_cls = field.model_cls
                if sub_model_cls is None:
                    raise TypeError(f"SubModel '{field_name}' is not configured with a model class")
                nested_value = getattr(model, field_name, None)
                if nested_value is None:
                    continue
                if not isinstance(nested_value, BaseModel):
                    raise TypeError(
                        f"Value for submodel '{field_name}' must be an instance of {sub_model_cls.__name__}"
                    )
                nested_key = f"{redis_key}:{field_name}"
                nested_data = nested_value.to_redis_hash()
                if nested_data:
                    self.redis.hset(nested_key, mapping=nested_data)
                # TTL preference: submodel Config.default_expire > collection expire
                sub_expire = getattr(getattr(sub_model_cls, "Config", object), "default_expire", None)
                expire = sub_expire if sub_expire is not None else self.expire
                if expire is not None:
                    self.redis.expire(nested_key, expire)

    def remove(self, key: K, cascade: bool = True) -> int:
        redis_key = self._make_key(key)
        deleted = self.redis.delete(redis_key)
        if cascade:
            # Delete nested submodel hashes
            cursor = 0
            pattern = f"{redis_key}:*"
            while True:
                cursor, keys = self.redis.scan(cursor=cursor, match=pattern)
                if keys:
                    self.redis.delete(*keys)
                if cursor == 0:
                    break
        return deleted

    def exists(self, key: K) -> bool:
        """Проверяет существование верхнеуровневого ключа коллекции в Redis."""
        redis_key = self._make_key(key)
        return bool(self.redis.exists(redis_key))

    def lock(self, key: K, ttl_ms: int) -> Lock:
        lock_key = f"{self.prefix}:_lock:{key}"
        return Lock(self.redis, lock_key, ttl_ms)

    def keys(self) -> Iterator[K]:
        """Возвращает генератор ключей в коллекции."""
        cursor = 0
        while True:
            cursor, keys = self.redis.scan(cursor=cursor, match=f"{self.prefix}:*")
            for key in keys:
                key_str = key.decode("utf-8")
                if ":_lock:" in key_str:
                    continue
                # Keep only top-level keys: exactly one ':' after prefix
                remainder = key_str[len(self.prefix) + 1 :]
                if ":" in remainder:
                    continue
                yield self.key_type(remainder)
            if cursor == 0:
                break

    def items(self) -> Iterator[tuple[K, T]]:
        """Возвращает генератор пар (ключ, объект) в коллекции."""
        for key in self.keys():
            yield key, self[key]
