# redis_storage/locking.py
import time
import uuid

class Lock:
    def __init__(self, redis, key: str, ttl_ms: int, sleep_ms: int = 100):
        self.redis = redis
        self.key = key
        self.ttl_ms = ttl_ms
        self.sleep_ms = sleep_ms
        self.token = str(uuid.uuid4())
        self.acquired = False

    def acquire(self, blocking: bool = True, timeout_ms: int | None = None):
        start = time.time()
        while True:
            if self.redis.set(self.key, self.token, nx=True, px=self.ttl_ms):
                self.acquired = True
                return True

            if not blocking:
                return False

            if timeout_ms is not None:
                elapsed = (time.time() - start) * 1000
                if elapsed > timeout_ms:
                    return False

            time.sleep(self.sleep_ms / 1000)

    def release(self):
        if not self.acquired:
            return
        # Упрощённый вариант (без Lua): если токен совпадает, снимаем
        value = self.redis.get(self.key)
        if value and value.decode() == self.token:
            self.redis.delete(self.key)
        self.acquired = False

    def __enter__(self):
        self.acquire()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()
