"""Basic typed field and model primitives used by the storage layer.

The Field classes act as descriptors. The BaseModel uses __init_subclass__
to automatically instantiate fields from type annotations.
"""

from __future__ import annotations

import json
from typing import Any, Dict, Generic, Optional, Type, TypeVar, get_args


T = TypeVar("T")
M = TypeVar("M", bound="BaseModel")


class Field(Generic[T]):
    def __init__(self):
        self.name: Optional[str] = None

    # Descriptor plumbing so IDEs know the instance attribute type is T
    def __set_name__(self, owner: type, name: str) -> None:
        self.name = name

    def __get__(self, instance: Any, owner: type | None = None) -> T | Field[T]:
        if instance is None:
            return self
        # store instance value in __dict__ under the field name
        return instance.__dict__.get(self.name)  # type: ignore[return-value]

    def __set__(self, instance: Any, value: T) -> None:
        instance.__dict__[self.name] = value

    # Serialization helpers used for Redis IO
    def serialize(self, value: T | None) -> Optional[str]:
        return str(value) if value is not None else None

    def deserialize(self, value: Any) -> T | None:
        return value


class Str(Field[str]):
    def deserialize(self, value: Any) -> Optional[str]:
        if value is None:
            return None
        return value.decode() if isinstance(value, bytes) else str(value)


class Int(Field[int]):
    def serialize(self, value: Optional[int]) -> Optional[str]:
        return str(int(value)) if value is not None else None

    def deserialize(self, value: Any) -> Optional[int]:
        return int(value) if value is not None else None


class Float(Field[float]):
    def serialize(self, value: Optional[float]) -> Optional[str]:
        return str(float(value)) if value is not None else None

    def deserialize(self, value: Any) -> Optional[float]:
        return float(value) if value is not None else None


class Bool(Field[bool]):
    def serialize(self, value: Optional[bool]) -> Optional[str]:
        if value is None:
            return None
        field_label = self.name or "Bool"
        if not isinstance(value, bool):
            raise TypeError(f"Bool field '{field_label}' expects a bool or None.")
        return "1" if value else "0"

    def deserialize(self, value: Any) -> Optional[bool]:
        if value is None:
            return None
        field_label = self.name or "Bool"
        if isinstance(value, bytes):
            value = value.decode()
        if isinstance(value, bool):
            return value
        if isinstance(value, int):
            if value in (0, 1):
                return bool(value)
        if isinstance(value, str):
            normalized = value.strip().lower()
            if normalized in ("1", "true"):
                return True
            if normalized in ("0", "false"):
                return False
        raise TypeError(f"Cannot deserialize value '{value}' to bool for field '{field_label}'.")


class List(Field[list[T]], Generic[T]):
    """Field for storing a list of typed items.

    New preferred usage: `my_list = List[Str]()` or `List[Int]()`.
    Backward compatible with the old annotated form: `my_list: List[Str] = List()`.
    """

    # Cache specialized List[T] classes so List[Int] is stable
    _specializations: dict[type[Field], type["List"]] = {}

    def __init__(self):
        super().__init__()
        self.inner_field: Optional[Field] = None

    def __set_name__(self, owner: type, name: str) -> None:
        super().__set_name__(owner, name)

        # If inner_field already configured (via List[T]()), do nothing
        if self.inner_field is not None:
            return

        # Backward compatibility: infer inner type from annotation `field: List[T] = List()`
        type_hint = owner.__annotations__.get(name)
        if not type_hint:
            # No annotation and not parameterized: cannot infer, leave helpful error
            raise TypeError(
                f"List field '{name}' in '{owner.__name__}' is not parameterized. "
                f"Use `List[T]()` or annotate as `{name}: List[T] = List()`"
            )

        args = get_args(type_hint)
        if not args:
            raise TypeError(f"List field '{name}' must specify an inner type. E.g., `List[Str]`")

        inner_type_cls = args[0]
        if isinstance(inner_type_cls, type) and issubclass(inner_type_cls, Field):
            self.inner_field = inner_type_cls()
        else:
            raise TypeError(
                f"Invalid inner type for List field '{name}'. It must be a Field subclass like Str or Int."
            )

    def serialize(self, value: Optional[list[T]]) -> Optional[str]:
        if self.inner_field is None:
            raise TypeError(
                "List field is not configured. Use `List[T]()` or provide a proper type annotation."
            )
        if value is None:
            return None
        serialized_list = [self.inner_field.serialize(item) for item in value]
        return json.dumps(serialized_list)

    def deserialize(self, value: Any) -> Optional[list[T]]:
        if self.inner_field is None:
            raise TypeError(
                "List field is not configured. Use `List[T]()` or provide a proper type annotation."
            )
        if value is None:
            return None
        if isinstance(value, bytes):
            value = value.decode('utf-8')
        deserialized_list = json.loads(value)
        return [self.inner_field.deserialize(item) for item in deserialized_list]

    def __class_getitem__(cls, item: Any) -> type["List"]:
        """Enable parameterization like List[Str] returning a specialized class.

        This returns a subclass of List that auto-configures `inner_field` to `item()`
        when instantiated, so `List[Str]()` works without a type annotation on the field.
        """
        if not (isinstance(item, type) and issubclass(item, Field)):
            raise TypeError("List[T] expects a Field subclass like Str or Int")

        # Return cached specialization if available
        cached = cls._specializations.get(item)
        if cached is not None:
            return cached

        inner_field_cls = item

        class _ListSpecialized(cls):  # type: ignore[misc]
            def __init__(self):
                super().__init__()
                self.inner_field = inner_field_cls()

        _ListSpecialized.__name__ = f"List[{inner_field_cls.__name__}]"
        cls._specializations[inner_field_cls] = _ListSpecialized
        return _ListSpecialized


class BaseModel:
    class Config:
        key_prefix: Optional[str] = None
        default_expire: Optional[int] = None

    def __init_subclass__(cls, **kwargs):
        """
        Эта магия вызывается при создании дочерних классов (User, Product).
        Она находит экземпляры полей, созданные из аннотаций, и регистрирует их.
        """
        super().__init_subclass__(**kwargs)
        # Создаем собственный, независимый словарь полей для каждого дочернего класса
        cls._fields: Dict[str, Field] = {}
        
        # Проходим по всем атрибутам класса, чтобы найти экземпляры полей
        for name, attr in cls.__dict__.items():
            if isinstance(attr, Field):
                # Мы нашли готовый экземпляр поля (дескриптор)!
                # Просто регистрируем его.
                cls._fields[name] = attr

    def __init__(self, **kwargs: Any):
        # Инициализируем значения полей из kwargs
        for field_name in self.__class__._fields:
            # Устанавливаем значение по умолчанию None, если оно не предоставлено
            setattr(self, field_name, kwargs.get(field_name))

    def to_redis_hash(self) -> Dict[str, str]:
        data: Dict[str, str] = {}
        # Используем кэшированный словарь _fields для итерации
        for field_name, field_instance in self.__class__._fields.items():
            value = getattr(self, field_name, None)
            serialized = field_instance.serialize(value)
            if serialized is not None:
                data[field_name] = serialized
        return data

    @classmethod
    def from_redis_hash(cls: type[M], data: Dict[str, Any]) -> M:
        kwargs: Dict[str, Any] = {}
        # Используем кэшированный словарь _fields для итерации
        for field_name, field_instance in cls._fields.items():
            raw_value = data.get(field_name)
            if raw_value is not None:
                kwargs[field_name] = field_instance.deserialize(raw_value)
        return cls(**kwargs)


class NestedModel(Field[M], Generic[M]):
    """Field to reference a nested BaseModel stored under a derived Redis key.

    Usage: `state = NestedModel[State]()` or annotated `state: NestedModel[State] = NestedModel()`.
    The field itself does not serialize into the parent hash. Nested model IO is
    handled by the Redis-backed proxy on access.
    """

    _specializations: dict[type[BaseModel], type["NestedModel"]] = {}

    def __init__(self):
        super().__init__()
        self.model_cls: Optional[Type[BaseModel]] = None

    def __set_name__(self, owner: type, name: str) -> None:
        super().__set_name__(owner, name)
        # Backward-compatible annotation-based configuration
        if self.model_cls is not None:
            return
        type_hint = owner.__annotations__.get(name)
        if not type_hint:
            # No annotation present; if user used SubModel[T](), model_cls is already set
            return
        args = get_args(type_hint)
        if not args:
            raise TypeError(f"SubModel field '{name}' must specify a model type, e.g. SubModel[State]")
        model_type = args[0]
        if isinstance(model_type, type) and issubclass(model_type, BaseModel):
            self.model_cls = model_type
        else:
            raise TypeError(
                f"Invalid model type for SubModel field '{name}'. It must be a BaseModel subclass."
            )

    def serialize(self, value: Optional[M]) -> Optional[str]:
        # Submodels are stored in separate Redis hashes, not as fields in parent hash
        return None

    def deserialize(self, value: Any) -> Optional[M]:
        # Not used; access handled via nested proxies
        return None

    def __class_getitem__(cls, item: Any) -> type["NestedModel"]:
        if not (isinstance(item, type) and issubclass(item, BaseModel)):
            raise TypeError("SubModel[T] expects a BaseModel subclass")

        cached = cls._specializations.get(item)
        if cached is not None:
            return cached

        model_cls = item

        class _SubModelSpecialized(cls):  # type: ignore[misc]
            def __init__(self):
                super().__init__()
                self.model_cls = model_cls

        _SubModelSpecialized.__name__ = f"SubModel[{model_cls.__name__}]"
        cls._specializations[model_cls] = _SubModelSpecialized
        return _SubModelSpecialized
